@extends('layouts.config_dashboard.app')

@section('style')
<style>
/* ============================================
   MULTIPAGE DOCUMENT VIEWER
   ============================================ */

.doc-viewer {
    display: flex;
    height: 85vh;
    background: #0a0a10;
    border-radius: 12px;
    overflow: hidden;
    border: 1px solid rgba(255,255,255,0.08);
}

/* LEFT SIDEBAR - Thumbnail Strip */
.thumb-sidebar {
    width: 140px;
    min-width: 140px;
    background: #12121a;
    border-right: 1px solid rgba(255,255,255,0.08);
    display: flex;
    flex-direction: column;
}

.thumb-header {
    padding: 12px;
    border-bottom: 1px solid rgba(255,255,255,0.08);
    font-size: 11px;
    color: #64748b;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    text-align: center;
}

.thumb-list {
    flex: 1;
    overflow-y: auto;
    padding: 8px;
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.thumb-item {
    position: relative;
    aspect-ratio: 3/4;
    border-radius: 6px;
    overflow: hidden;
    cursor: pointer;
    border: 2px solid transparent;
    transition: all 0.2s;
    background: #1a1a24;
}

.thumb-item:hover {
    border-color: rgba(16, 185, 129, 0.5);
}

.thumb-item.active {
    border-color: #10b981;
    box-shadow: 0 0 12px rgba(16, 185, 129, 0.3);
}

.thumb-item img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.thumb-number {
    position: absolute;
    bottom: 4px;
    right: 4px;
    background: rgba(0,0,0,0.7);
    color: #fff;
    font-size: 10px;
    padding: 2px 6px;
    border-radius: 4px;
}

.thumb-type {
    position: absolute;
    top: 4px;
    left: 4px;
    background: rgba(16,185,129,0.8);
    color: #fff;
    font-size: 8px;
    padding: 2px 5px;
    border-radius: 3px;
    max-width: 90%;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

/* MAIN AREA */
.main-area {
    flex: 1;
    display: flex;
    flex-direction: column;
    min-width: 0;
}

/* HEADER */
.viewer-header {
    background: #16161f;
    padding: 10px 16px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    border-bottom: 1px solid rgba(255,255,255,0.06);
    gap: 12px;
    flex-wrap: wrap;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 12px;
}

.header-left .icon {
    width: 36px;
    height: 36px;
    background: rgba(16,185,129,0.15);
    color: #10b981;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 8px;
    font-size: 18px;
}

.header-left h5 {
    margin: 0;
    font-size: 14px;
    color: #e2e8f0;
}

.header-left .meta {
    font-size: 11px;
    color: #64748b;
}

/* Controls */
.controls {
    display: flex;
    align-items: center;
    gap: 10px;
}

.ctrl-group {
    display: flex;
    background: rgba(255,255,255,0.05);
    border-radius: 8px;
    padding: 3px;
    gap: 2px;
}

.ctrl-btn {
    width: 32px;
    height: 32px;
    border: none;
    background: transparent;
    color: #94a3b8;
    border-radius: 6px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
    transition: all 0.15s;
}

.ctrl-btn:hover { background: rgba(255,255,255,0.1); color: #fff; }
.ctrl-btn.active { background: #10b981; color: #fff; }

.ctrl-label {
    padding: 0 8px;
    font-size: 12px;
    color: #cbd5e1;
    display: flex;
    align-items: center;
    min-width: 45px;
    justify-content: center;
}

.sep { width: 1px; height: 24px; background: rgba(255,255,255,0.1); margin: 0 4px; }

.action-btn {
    padding: 7px 14px;
    border-radius: 6px;
    border: none;
    font-size: 13px;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 6px;
    transition: all 0.15s;
    text-decoration: none;
}

.btn-ghost { background: rgba(255,255,255,0.05); color: #cbd5e1; }
.btn-ghost:hover { background: rgba(255,255,255,0.12); color: #fff; }
.btn-close { background: rgba(239,68,68,0.1); color: #f87171; }
.btn-close:hover { background: #ef4444; color: #fff; }

/* CANVAS STAGE */
.canvas-stage {
    flex: 1;
    position: relative;
    background: radial-gradient(circle at center, #1a1a24 0%, #0a0a10 100%);
    overflow: hidden;
    cursor: grab;
}

.canvas-stage:active { cursor: grabbing; }

.image-container {
    position: absolute;
    top: 50%;
    left: 50%;
    transform-origin: center center;
    will-change: transform;
}

.main-image {
    display: block;
    box-shadow: 0 20px 60px rgba(0,0,0,0.6);
    pointer-events: none;
    border-radius: 2px;
}

/* LOADER */
.loader {
    position: absolute;
    inset: 0;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    gap: 12px;
    background: rgba(10,10,16,0.85);
    z-index: 10;
    transition: opacity 0.3s;
}

.loader.hidden { opacity: 0; pointer-events: none; }

.spinner {
    width: 36px;
    height: 36px;
    border: 3px solid rgba(255,255,255,0.1);
    border-top-color: #10b981;
    border-radius: 50%;
    animation: spin 0.7s linear infinite;
}

@keyframes spin { to { transform: rotate(360deg); } }

/* BOTTOM NAV */
.bottom-nav {
    background: #16161f;
    padding: 8px 16px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-top: 1px solid rgba(255,255,255,0.06);
}

.page-nav {
    display: flex;
    align-items: center;
    gap: 8px;
}

.nav-btn {
    width: 32px;
    height: 32px;
    border: none;
    background: rgba(255,255,255,0.05);
    color: #94a3b8;
    border-radius: 6px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
    transition: all 0.15s;
}

.nav-btn:hover:not(:disabled) { background: rgba(255,255,255,0.12); color: #fff; }
.nav-btn:disabled { opacity: 0.3; cursor: not-allowed; }

.page-indicator {
    font-size: 13px;
    color: #94a3b8;
    min-width: 80px;
    text-align: center;
}

.page-indicator strong {
    color: #fff;
}

.status-text {
    font-size: 11px;
    color: #64748b;
}

/* FULLSCREEN */
:fullscreen .doc-viewer {
    height: 100vh;
    border-radius: 0;
    border: none;
}

/* RESPONSIVE */
@media (max-width: 768px) {
    .thumb-sidebar { width: 80px; min-width: 80px; }
    .thumb-header { display: none; }
    .viewer-header { padding: 8px 12px; }
    .action-btn span { display: none; }
}
</style>
@endsection

@section('content')
<div class="container-fluid mt-3">
    <div class="doc-viewer" id="docViewer">
        
        <!-- LEFT SIDEBAR - Thumbnails -->
        <div class="thumb-sidebar">
            <div class="thumb-header">Pages ({{ $allDocuments->count() }})</div>
            <div class="thumb-list" id="thumbList">
                @foreach($allDocuments as $index => $doc)
                <div class="thumb-item {{ $doc->id == $document->id ? 'active' : '' }}" 
                     data-id="{{ $doc->id }}"
                     data-index="{{ $index }}">
                    <img src="{{ Storage::disk('s3')->temporaryUrl($doc->path, now()->addMinutes(30)) }}" 
                         alt="Page {{ $index + 1 }}"
                         loading="lazy">
                    <span class="thumb-number">{{ $index + 1 }}</span>
                    <span class="thumb-type">{{ $doc->typedocument->type ?? 'Doc' }}</span>
                </div>
                @endforeach
            </div>
        </div>

        <!-- MAIN AREA -->
        <div class="main-area">
            
            <!-- HEADER -->
            <div class="viewer-header">
                <div class="header-left">
                    <div class="icon"><i class="mdi mdi-file-document-outline"></i></div>
                    <div>
                        <h5 id="docTitle">{{ $document->typedocument->type ?? 'Document' }}</h5>
                        <div class="meta">Dossier: {{ $document->dossier->ref ?? 'N/A' }}</div>
                    </div>
                </div>

                <div class="controls">
                    <!-- Zoom -->
                    <div class="ctrl-group">
                        <button class="ctrl-btn" id="zoomOut" title="Zoom -"><i class="mdi mdi-minus"></i></button>
                        <span class="ctrl-label" id="zoomLabel">100%</span>
                        <button class="ctrl-btn" id="zoomIn" title="Zoom +"><i class="mdi mdi-plus"></i></button>
                        <button class="ctrl-btn" id="fitBtn" title="Ajuster (F)"><i class="mdi mdi-fit-to-screen"></i></button>
                    </div>

                    <div class="sep"></div>

                    <!-- Rotate -->
                    <div class="ctrl-group">
                        <button class="ctrl-btn" id="rotateL" title="Gauche (L)"><i class="mdi mdi-rotate-left"></i></button>
                        <span class="ctrl-label" id="rotateLabel">0°</span>
                        <button class="ctrl-btn" id="rotateR" title="Droite (R)"><i class="mdi mdi-rotate-right"></i></button>
                    </div>

                    <div class="sep"></div>

                    <!-- Actions -->
                    <button class="action-btn btn-ghost" id="downloadBtn"><i class="mdi mdi-download"></i><span>Télécharger</span></button>
                    <button class="action-btn btn-ghost" id="printBtn"><i class="mdi mdi-printer"></i></button>
                    <button class="action-btn btn-ghost" id="fullscreenBtn"><i class="mdi mdi-fullscreen"></i></button>
                    <a href="{{ url()->previous() }}" class="action-btn btn-close"><i class="mdi mdi-close"></i></a>
                </div>
            </div>

            <!-- CANVAS -->
            <div class="canvas-stage" id="stage">
                <div class="loader" id="loader">
                    <div class="spinner"></div>
                    <span style="color:#94a3b8">Chargement...</span>
                </div>

                <div class="image-container" id="imageContainer">
                    <img id="mainImage" 
                         class="main-image"
                         src="{{ Storage::disk('s3')->temporaryUrl($document->path, now()->addMinutes(30)) }}"
                         alt="Document"
                         draggable="false">
                </div>
            </div>

            <!-- BOTTOM NAV -->
            <div class="bottom-nav">
                <div class="page-nav">
                    <button class="nav-btn" id="prevPage" title="Page précédente"><i class="mdi mdi-chevron-left"></i></button>
                    <span class="page-indicator"><strong id="currentPage">1</strong> / <span id="totalPages">{{ $allDocuments->count() }}</span></span>
                    <button class="nav-btn" id="nextPage" title="Page suivante"><i class="mdi mdi-chevron-right"></i></button>
                </div>
                <div class="status-text" id="statusText">Prêt</div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
document.addEventListener('DOMContentLoaded', () => {
    // === DATA ===
    const pages = @json($allDocuments->map(fn($d) => [
        'id' => $d->id,
        'url' => Storage::disk('s3')->temporaryUrl($d->path, now()->addMinutes(30)),
        'type' => $d->typedocument->type ?? 'Document'
    ])->values());

    const initialIndex = pages.findIndex(p => p.id === {{ $document->id }});

    // === STATE ===
    const state = {
        currentIndex: initialIndex >= 0 ? initialIndex : 0,
        x: 0,
        y: 0,
        scale: 1,
        rotation: 0,
        isDragging: false,
        startX: 0,
        startY: 0,
        initX: 0,
        initY: 0,
        naturalW: 0,
        naturalH: 0
    };

    // === DOM ===
    const stage = document.getElementById('stage');
    const container = document.getElementById('imageContainer');
    const img = document.getElementById('mainImage');
    const loader = document.getElementById('loader');
    const zoomLabel = document.getElementById('zoomLabel');
    const rotateLabel = document.getElementById('rotateLabel');
    const currentPageEl = document.getElementById('currentPage');
    const statusText = document.getElementById('statusText');
    const docTitle = document.getElementById('docTitle');
    const thumbItems = document.querySelectorAll('.thumb-item');

    // === RENDER ===
    function render() {
        container.style.transform = `translate(calc(-50% + ${state.x}px), calc(-50% + ${state.y}px)) rotate(${state.rotation}deg) scale(${state.scale})`;
        zoomLabel.textContent = Math.round(state.scale * 100) + '%';
        rotateLabel.textContent = state.rotation + '°';
    }

    // === FIT ===
    function fit() {
        if (!state.naturalW) return;
        const rect = stage.getBoundingClientRect();
        const w = rect.width - 60;
        const h = rect.height - 60;
        const isVert = state.rotation % 180 !== 0;
        const cw = isVert ? state.naturalH : state.naturalW;
        const ch = isVert ? state.naturalW : state.naturalH;
        state.scale = Math.min(w / cw, h / ch, 1);
        state.x = 0;
        state.y = 0;
        render();
    }

    // === LOAD PAGE ===
    function loadPage(index) {
        if (index < 0 || index >= pages.length) return;
        
        state.currentIndex = index;
        state.x = 0;
        state.y = 0;
        state.rotation = 0;
        
        loader.classList.remove('hidden');
        img.src = pages[index].url;
        
        // Update UI
        currentPageEl.textContent = index + 1;
        docTitle.textContent = pages[index].type;
        document.getElementById('prevPage').disabled = index === 0;
        document.getElementById('nextPage').disabled = index === pages.length - 1;
        
        // Update thumbnails
        thumbItems.forEach((t, i) => t.classList.toggle('active', i === index));
        
        // Scroll thumbnail into view
        const activeThumb = document.querySelector('.thumb-item.active');
        if (activeThumb) activeThumb.scrollIntoView({ block: 'nearest', behavior: 'smooth' });
    }

    // === IMAGE LOAD ===
    img.onload = () => {
        state.naturalW = img.naturalWidth;
        state.naturalH = img.naturalHeight;
        loader.classList.add('hidden');
        fit();
        statusText.textContent = `${state.naturalW} × ${state.naturalH} px`;
    };

    img.onerror = () => {
        loader.innerHTML = '<span style="color:#ef4444">Erreur de chargement</span>';
    };

    // === DRAG/PAN ===
    stage.addEventListener('mousedown', e => {
        if (e.button !== 0) return;
        state.isDragging = true;
        state.startX = e.clientX;
        state.startY = e.clientY;
        state.initX = state.x;
        state.initY = state.y;
        stage.style.cursor = 'grabbing';
        e.preventDefault();
    });

    window.addEventListener('mousemove', e => {
        if (!state.isDragging) return;
        state.x = state.initX + (e.clientX - state.startX);
        state.y = state.initY + (e.clientY - state.startY);
        render();
    });

    window.addEventListener('mouseup', () => {
        if (state.isDragging) {
            state.isDragging = false;
            stage.style.cursor = 'grab';
        }
    });

    // === WHEEL ZOOM ===
    stage.addEventListener('wheel', e => {
        e.preventDefault();
        const delta = e.deltaY * -0.001;
        state.scale = Math.min(Math.max(0.1, state.scale + delta), 8);
        render();
    });

    // === CONTROLS ===
    document.getElementById('zoomIn').onclick = () => { state.scale = Math.min(state.scale * 1.25, 8); render(); };
    document.getElementById('zoomOut').onclick = () => { state.scale = Math.max(state.scale / 1.25, 0.1); render(); };
    document.getElementById('fitBtn').onclick = fit;
    document.getElementById('rotateL').onclick = () => { state.rotation = (state.rotation - 90) % 360; render(); };
    document.getElementById('rotateR').onclick = () => { state.rotation = (state.rotation + 90) % 360; render(); };

    // === PAGE NAV ===
    document.getElementById('prevPage').onclick = () => loadPage(state.currentIndex - 1);
    document.getElementById('nextPage').onclick = () => loadPage(state.currentIndex + 1);

    // === THUMBNAIL CLICKS ===
    thumbItems.forEach((t, i) => {
        t.addEventListener('click', () => loadPage(i));
    });

    // === KEYBOARD ===
    document.addEventListener('keydown', e => {
        if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') return;
        switch (e.key) {
            case 'ArrowLeft': loadPage(state.currentIndex - 1); break;
            case 'ArrowRight': loadPage(state.currentIndex + 1); break;
            case 'ArrowUp': e.preventDefault(); state.scale = Math.min(state.scale * 1.2, 8); render(); break;
            case 'ArrowDown': e.preventDefault(); state.scale = Math.max(state.scale / 1.2, 0.1); render(); break;
            case 'r': case 'R': state.rotation = (state.rotation + 90) % 360; render(); break;
            case 'l': case 'L': state.rotation = (state.rotation - 90) % 360; render(); break;
            case 'f': case 'F': fit(); break;
            case '0': state.scale = 1; state.x = 0; state.y = 0; state.rotation = 0; render(); break;
        }
    });

    // === ACTIONS ===
    document.getElementById('downloadBtn').onclick = () => {
        const a = document.createElement('a');
        a.href = pages[state.currentIndex].url;
        a.download = `document_page_${state.currentIndex + 1}`;
        a.click();
    };

    document.getElementById('printBtn').onclick = () => {
        const w = window.open('', '_blank');
        w.document.write(`<img src="${pages[state.currentIndex].url}" style="max-width:100%" onload="window.print();window.close()">`);
        w.document.close();
    };

    document.getElementById('fullscreenBtn').onclick = () => {
        const el = document.getElementById('docViewer');
        if (!document.fullscreenElement) el.requestFullscreen();
        else document.exitFullscreen();
    };

    // === INIT ===
    loadPage(state.currentIndex);
});
</script>
@endsection