<?php

namespace App\Services;

use App\Models\Demande;
use App\Models\Dossier;
use App\Models\Rendezvous;
use App\Models\Seuil;
use App\Models\Ticket;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class SidebarCache
{
    const CACHE_TTL = 300; // 5 minutes

    /**
     * Get cache key for current user
     * Each user has their own cache based on permissions
     */
    private static function getCacheKey()
    {
        $userId = Auth::id() ?? 0;
        return "sidebar_counters_user_{$userId}";
    }

    /**
     * Get all counters for current user
     */
    public static function getAll()
    {
        if (!Auth::check()) {
            return self::getEmptyCounters();
        }

        return Cache::remember(self::getCacheKey(), self::CACHE_TTL, function() {
            return self::recalculate();
        });
    }

    /**
     * Clear cache for current user
     */
    public static function clear()
    {
        if (Auth::check()) {
            Cache::forget(self::getCacheKey());
        }
    }

    /**
     * Clear cache for a specific user
     */
    public static function clearForUser($userId)
    {
        Cache::forget("sidebar_counters_user_{$userId}");
    }

    /**
     * Clear cache for ALL users (use when global data changes)
     * Call this from admin actions that affect multiple users
     */
    public static function clearAll()
    {
        // Get all user IDs and clear their caches
        $userIds = \App\Models\User::pluck('id');
        foreach ($userIds as $userId) {
            Cache::forget("sidebar_counters_user_{$userId}");
        }
    }

    /**
     * Return empty counters array
     */
    private static function getEmptyCounters()
    {
        return [
            'demandes' => 0, 'rdv' => 0, 'ticketcount' => 0, 'ticketcountterminer' => 0,
            'dossiersfacture' => 0, 'tribunale' => 0, 'douteux' => 0, 'rapide' => 0,
            'docasigner' => 0, 'docsigner' => 0, 'aexpcount' => 0, 'enexp' => 0,
            'attdevis' => 0, 'attexpap' => 0, 'attfacture' => 0, 'attrapport' => 0,
            'basedoc' => 0, 'deuxdevis' => 0, 'senddevis' => 0, 'avisdevis' => 0,
            'traitdevis' => 0, 'count' => 0, 'controldevis' => 0, 'attdevisav' => 0,
            'deuxfacture' => 0, 'avisfacture' => 0, 'traitfacture' => 0, 'count2' => 0,
            'attfactureap' => 0, 'sendfacture' => 0, 'deuxrapport' => 0, 'sendrapport' => 0,
            'rapportasigner' => 0, 'count3' => 0
        ];
    }

    /**
     * Recalculate all counters for current user
     * Uses Dossier::user() scope to respect permissions
     */
    public static function recalculate()
    {
        $user = Auth::user();
        if (!$user) {
            return self::getEmptyCounters();
        }

        $counts = [];
        $roleId = $user->role_id;

        // ========== TICKETS (User-specific) ==========
        $counts['ticketcount'] = Ticket::where('user_id', $user->id)
            ->where('status_rendezvous_id', 2)
            ->count();
            
        $counts['ticketcountterminer'] = Ticket::where('creator_id', $user->id)
            ->where('status_rendezvous_id', 3)
            ->count();

        // ========== DEMANDES & RDV (Global - based on permission) ==========
        $counts['demandes'] = Demande::where('status_rendezvous_id', 2)->count();
        $counts['rdv'] = Rendezvous::where('status_rendezvous_id', 2)->count();

        // ========== DOSSIERS FACTURE (Role 3 & 4 only) ==========
        if ($roleId == 3 || $roleId == 4) {
            $counts['dossiersfacture'] = Dossier::whereNull('facture')
                ->where('dossier_status_id', 2)
                ->where('update', 1)
                ->count();
        } else {
            $counts['dossiersfacture'] = 0;
        }

        // ========== DOSSIER TYPES (Using user() scope) ==========
        $counts['tribunale'] = Dossier::user()
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->where('type_expertise_id', 9)
            ->count();

        $counts['douteux'] = Dossier::user()
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->where('sinistre_douteux', 1)
            ->count();

        // Rapide: Role 2 has different filter
        if ($roleId == 2) {
            $counts['rapide'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->where('type_expertise_id', 16)
                ->whereIn('step_id', [4, 9])
                ->count();
        } elseif ($roleId != 3 && $roleId != 6) {
            $counts['rapide'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->where('type_expertise_id', 16)
                ->count();
        } else {
            $counts['rapide'] = 0;
        }

        // ========== DOCUMENT COUNTS (Role 1 & 4 only) ==========
        if ($roleId == 1 || $roleId == 4) {
            $counts['docasigner'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereHas('document', function ($query) {
                    $query->where('type_document_id', 21)
                        ->where('signer', 0)
                        ->where('rejected', 0);
                })->count();

            $counts['docsigner'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereHas('document', function ($query) {
                    $query->where('type_document_id', 21)
                        ->where('signer', 1)
                        ->where('rejected', 0);
                })
                ->whereDoesntHave('onerapport', function ($query) {
                    $query->whereIn('rapport_status_id', [2, 3]);
                })->count();
        } else {
            $counts['docasigner'] = 0;
            $counts['docsigner'] = 0;
        }

        // ========== EXPERTISE/CARENCE COUNTS (Role 1 & 4) ==========
        if ($roleId == 1 || $roleId == 4) {
            // Attente Photos Avant
            $counts['aexpcount'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereDoesntHave('devis')
                ->whereDoesntHave('document', function ($query) {
                    $query->where('type_document_id', 5);
                })->count();

            // En Expertise
            $counts['enexp'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereHas('meeting', function ($query) {
                    $query->where('meeting_status_id', 2);
                })->count();

            // Attente Documents Base
            $counts['basedoc'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereDoesntHave('document', function ($query) {
                    $query->whereIn('type_document_id', [1, 2, 3]);
                })->count();

            // Attente Devis
            $counts['attdevis'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereDoesntHave('devis')
                ->count();

            // Attente Photos Après
            $counts['attexpap'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->where('type_expertise_id', '!=', 14)
                ->where('mode_id', '!=', 7)
                ->whereDoesntHave('document', function ($query) {
                    $query->where('type_document_id', 6);
                })
                ->whereHas('onedevis', function ($query) {
                    $query->where('devis_status_id', 2)
                        ->orWhere('devis_status_id', 7);
                })
                ->whereDoesntHave('facture')
                ->count();

            // Attente Facture
            $counts['attfacture'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->where('type_expertise_id', '!=', 14)
                ->where('mode_id', '!=', 7)
                ->whereHas('document', function ($query) {
                    $query->where('type_document_id', 6);
                })
                ->whereHas('onedevis', function ($query) {
                    $query->where('devis_status_id', 2)
                        ->orWhere('devis_status_id', 7);
                })
                ->whereDoesntHave('facture')
                ->count();

            // Attente Rapport
            $counts['attrapport'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereHas('devis', function ($query) {
                    $query->where('devis_status_id', 2)
                        ->orWhere('devis_status_id', 7);
                })
                ->whereHas('facture', function ($query) {
                    $query->where('devis_status_id', 2)
                        ->orWhere('devis_status_id', 7);
                })
                ->whereDoesntHave('rapport')
                ->count();
        } else {
            $counts['aexpcount'] = 0;
            $counts['enexp'] = 0;
            $counts['basedoc'] = 0;
            $counts['attdevis'] = 0;
            $counts['attexpap'] = 0;
            $counts['attfacture'] = 0;
            $counts['attrapport'] = 0;
        }

        // ========== DEVIS STATUS COUNTS ==========
        $counts['deuxdevis'] = 0;
        $counts['senddevis'] = 0;
        $counts['avisdevis'] = 0;
        $counts['traitdevis'] = 0;
        $counts['count'] = 0;  // Devis à retraiter
        $counts['controldevis'] = 0;
        $counts['attdevisav'] = 0;

        // Get dossiers with devis using user() scope
        $dossiersWithDevis = Dossier::user()
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->whereHas('devis')
            ->with(['onedevis', 'montant'])
            ->get();

        foreach ($dossiersWithDevis as $dossier) {
            if (!$dossier->onedevis) continue;
            
            $statusId = $dossier->onedevis->devis_status_id;

            switch ($statusId) {
                case 1: // Attente 2ème Devis
                    if ($roleId == 1 || $roleId == 4) {
                        $counts['deuxdevis']++;
                    }
                    break;
                    
                case 2: // Accord Devis
                    if ($roleId == 1 || $roleId == 4) {
                        $counts['senddevis']++;
                    }
                    break;
                    
                case 3: // Demande Avis Devis
                    $counts['avisdevis']++;
                    break;
                    
                case 4: // Devis à Traiter / Attente Accord
                    if ($roleId == 2 || $roleId == 4) {
                        // Role 2: Check seuil
                        if ($roleId == 2) {
                            $seuil = Seuil::where('company_id', $dossier->company_id)
                                ->where('user_id', $user->id)
                                ->first();
                            if ($seuil && $dossier->onedevis->montant_devis > $seuil->min 
                                && $dossier->onedevis->montant_devis < $seuil->max) {
                                $counts['traitdevis']++;
                            }
                        } else {
                            // Role 4: Check mode and valeurs
                            if ($dossier->mode_id == 3 || $dossier->mode_id == 2) {
                                $counts['traitdevis']++;
                            } else {
                                $counts['traitdevis']++;
                            }
                        }
                    }
                    // Also count for Role 1 & 4 as "Devis Attente Accord"
                    if ($roleId == 1 || $roleId == 4) {
                        // This is already counted in traitdevis for display
                    }
                    break;
                    
                case 5: // Devis à Retraiter
                    if ($roleId == 2) {
                        $seuil = Seuil::where('company_id', $dossier->company_id)
                            ->where('user_id', $user->id)
                            ->first();
                        if ($seuil && $dossier->onedevis->montant_devis > $seuil->min 
                            && $dossier->onedevis->montant_devis < $seuil->max) {
                            $counts['count']++;
                        }
                    } elseif ($roleId == 4) {
                        $counts['count']++;
                    }
                    break;
                    
                case 8: // Devis à Controler
                    if ($roleId == 2) {
                        $seuil = Seuil::where('company_id', $dossier->company_id)
                            ->where('user_id', $user->id)
                            ->first();
                        if ($seuil && $dossier->montant && $dossier->montant->montant_accord < $seuil->accord) {
                            $counts['controldevis']++;
                        }
                    } elseif ($roleId == 4) {
                        $counts['controldevis']++;
                    }
                    break;
                    
                case 9: // Devis Sans Photo
                    if ($roleId == 1 || $roleId == 4) {
                        $counts['attdevisav']++;
                    }
                    break;
            }
        }

        // ========== FACTURE STATUS COUNTS ==========
        $counts['deuxfacture'] = 0;
        $counts['avisfacture'] = 0;
        $counts['traitfacture'] = 0;
        $counts['count2'] = 0;  // Facture à retraiter
        $counts['attfactureap'] = 0;
        $counts['sendfacture'] = 0;

        $dossiersWithFacture = Dossier::user()
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->whereHas('facture')
            ->with('onefacture')
            ->get();

        foreach ($dossiersWithFacture as $dossier) {
            if (!$dossier->onefacture) continue;
            
            $statusId = $dossier->onefacture->devis_status_id;

            switch ($statusId) {
                case 1: // Attente 2ème Facture
                    if ($roleId == 1 || $roleId == 4) {
                        $counts['deuxfacture']++;
                    }
                    break;
                    
                case 3: // Demande Avis Facture
                    $counts['avisfacture']++;
                    break;
                    
                case 4: // Facture à Traiter
                    if ($roleId == 2) {
                        $seuil = Seuil::where('company_id', $dossier->company_id)
                            ->where('user_id', $user->id)
                            ->first();
                        if ($seuil && $dossier->onefacture->montant_facture > $seuil->min 
                            && $dossier->onefacture->montant_facture < $seuil->max) {
                            $counts['traitfacture']++;
                        }
                    } elseif ($roleId == 4) {
                        $counts['traitfacture']++;
                    }
                    break;
                    
                case 5: // Facture à Retraiter
                    if ($roleId == 2) {
                        $seuil = Seuil::where('company_id', $dossier->company_id)
                            ->where('user_id', $user->id)
                            ->first();
                        if ($seuil && $dossier->onefacture->montant_facture > $seuil->min 
                            && $dossier->onefacture->montant_facture < $seuil->max) {
                            $counts['count2']++;
                        }
                    } elseif ($roleId == 4) {
                        $counts['count2']++;
                    }
                    break;
                    
                case 9: // Facture Sans Photo
                    if ($roleId == 1 || $roleId == 4) {
                        $counts['attfactureap']++;
                    }
                    break;
            }
        }

        // Sendfacture - Accord Facture (no rapport, facture status 2)
        if ($roleId == 1 || $roleId == 4) {
            $counts['sendfacture'] = Dossier::user()
                ->where('dossier_status_id', 2)
                ->where('update', 0)
                ->whereDoesntHave('rapport')
                ->whereHas('facture', function ($query) {
                    $query->where('devis_status_id', 2);
                })->count();
        }

        // ========== RAPPORT STATUS COUNTS ==========
        $counts['deuxrapport'] = 0;
        $counts['sendrapport'] = 0;
        $counts['rapportasigner'] = 0;
        $counts['count3'] = 0;  // Instance Rapport

        $dossiersWithRapport = Dossier::user()
            ->where('dossier_status_id', 2)
            ->whereHas('rapport')
            ->with('onerapport')
            ->get();

        foreach ($dossiersWithRapport as $dossier) {
            if (!$dossier->onerapport) continue;
            
            $statusId = $dossier->onerapport->rapport_status_id;

            // Sendrapport counts ALL with status 2 (no update check)
            if ($statusId === 2) {
                if ($roleId == 1 || $roleId == 4) {
                    $counts['sendrapport']++;
                }
            }

            // Others only count if update == 0
            if ($dossier->update == 0) {
                switch ($statusId) {
                    case 1: // Rapport Refusé
                        if ($roleId == 1 || $roleId == 4) {
                            $counts['deuxrapport']++;
                        }
                        break;
                        
                    case 4: // Rapport à Signer
                        $counts['rapportasigner']++;
                        break;
                        
                    case 5: // Instance Rapport
                        if ($roleId == 1 || $roleId == 4) {
                            $counts['count3']++;
                        }
                        break;
                }
            }
        }

        return $counts;
    }
}